function [PDF,MSD_results,PDF_results,AC_results, parametersMSD, parameters1s, parameters2s, parameters3s, parameters4s, AIC, parametersAC] = analysis_modified(traj,xposure,magnif)
%function [PDF,MSD_results,PDF_results,AC_results, parametersMSD, parameters1s, parameters2s, parameters3s, parameters4s, AIC, parametersAC] = analysis_modified(traj,xposure,magnif)
%
%This program was originally written by Matsuoka, Shibata, and Ueda for the
%paper Biophys J, 97, 2009, 1115-1124, Stat anal of lateral diff and
%multistate kin in SMT.  In it's original form, the first part of the
%program simulates single molecule motion of the user defined number of
%trajectories and steps, with diffusion coefficients D1 and D2 if desired.
%The switching of states can occur between trajectories (different
%populations) or within trajectories (intermittent diffusion). The second
%part of the program analyzes that motion.
%I inserted a new option- to read in experimental data, which is then
%analyzed the same way.
%
%Original authors: Matsuoka, Shibata, and Ueda
%Modified to accept real trajectories by Lindsay Elliott, UIUC and ND, 5/27/2010
%of  various lengths
%
%INPUTS(OPTIONAL):  
%           TRAJ:       This array contains all the information learned
%                       about all the particles, stored in columns. A
%                       single trajectory may occupy many sequential rows.
%                       A single row details information about a feature in
%                       a particular frame, or that features trajectory
%                       from the previous frame to that frame.
%           TRAJ(:,1)   The x centroid positions, in pixels
%           TRAJ(:,2)   The y centroid positions, in pixels
%           TRAJ(:,3)   The integrated brightness of the features,
%                       background corrected.
%           TRAJ(:,4)   The radiui of gyration squared of the features
%           TRAJ(:,5)   The eccentricity of the features
%           TRAJ(:,6)   The frame number (real frame number) of this row.
%           TRAJ(:,7)   The unique particle ID of this particle
%           TRAJ(:,8)   How many frames this particle has been tracked so
%                       far.
%           TRAJ(:,9)   The x displacement of the particle since its last
%                       position. Note, this will include -1000 for any
%                       frames where the position of the particle in the
%                       previous frame is unknown.
%           TRAJ(:,10)  The y displacement of the particle since its last
%                       position. Note, this will include -1000 for any
%                       frames where the position of the particle in the
%                       previous frame is unknown.
%           TRAJ(:,11)  The squared displacement of the particle to this
%                       point.
%           TRAJ(:,12)  Step size in microns, earlier in the program used
%                       to form trajectories:
%                       The presplicing alternating frame tracking unique
%                       particle ID. Even and odd sequences are
%                       differentiated throught the use of real and complex.
%           TRAJ(:,13)  The presplicing single frame tracking unique
%                       particle ID.
%           XPOSURE:    Kinetic cycle time in milliseconds, will be
%                       converted to seconds in this program.
%           MAGNIF:     The magnification the experiment was taken at (the
%                       magnification of the microscope generally).
%
%OUTPUTS:   
%       PDF            Data for the probability distribution function of
%                      displacement in microns, bin size set by user (see
%                      line below starting: dr = )
%       MSD_results    Mean squared displacement calculation for the first
%                      ten time delay points, column 1 is time delay in
%                      seconds, column 2 is the calculated MSD in microns,
%                      and column 3 is the fitted function at those points
%                      also in microns
%       PDF_results    Fits for 1, 2, 3, and 4 states of the data from PDF
%                      in columns 2-5, column 1 is the bin center
%       AC_results     Autocorrelation function results for squared
%                      displacement, column 1 is time delay in seconds,
%                      column 2 is the calculated data, column 3 is the
%                      fitted function to that data
%       parametersMSD  Slope and intercept from the MSD fit
%       parameters1s   Diffusion coefficient (um2/s) for 1 state fit of the step size PDF
%       parameters2s   Diffusion coefficients (um2/s) and fractions of the
%                      population for 2 state fit ""
%       parameters3s   Same for 3 states, fraction of population is 
%                       x/(sum all x)
%       parameters4s   Same for 4 states
%       AIC            Akaike information criterion results for 1, 2, 3,
%                      and 4 states
%       parametersAC   Parameters for the fit of squared displacement
%                      autocorrelation, a, K, b for: F=a*exp(-K*xdata)+b

%% simulation of Brownian motion and state transition, if traj is not
%% inputted, trajectories can be simulated with the dt, F, T, X, D1, D2,
%% k12, k21, p, q, sigma chosen by the user
if nargin == 0
    disp('data was simulated in program');
    % trajectory
    dt=1/30; %[sec]% time interval between two neighboring positions
    F=200; % number of time steps
    T(1, :)=0:dt:dt*F; %[sec]% time along the trajectory
    X=100; % number of trajectories
    
    % molecular movement
    D1=0.1; %[micrometer^2/sec]% diffusion coefficient for state 1
    D2=0.5; %[micrometer^2/sec]% diffusion coefficient for state 2
    k12=3; %[/sec]% rate constant for transition from state 1 to state 2
    k21=1; %[/sec]% rate constant for transition from state 2 to state 1
    p=0.25; % proportion of molecules which adopt state 1 
    % p equals to k21/(k12+k21) in equilibrium
    q=1-p; % proportion of molecules which adopt state 2
    
    % measurement error
    sigma=0.04; %[micrometer]% standard deviation of the error
    sigma2=sigma^2;

    % simulation
    rand('twister', 0)
    A={1,X}; % trajectories of X molecules, each cell contains a 2 column vector
             % with x and y position data in microns
    for i=1:X  
        R=zeros(F,2); % x- and y-coordinates
        Rapp=zeros(F,2); % x- and y-coordinates after the error incorporated
        Rapp(1, :)=R(1, :)+sigma*randn(1, 2);
        S=zeros(F,2); % time trajectory of the state
        k=1; %% consider an initial state
        if p>rand 
            S(1, 1)=1; % state=1
        else          
            S(1, 1)=2; % state=2
        end
                
        for k=2:F       
            if S(k-1, 1)==1;
                R(k, :)=R(k-1, :)+sqrt(2.0*D1*dt)*randn(1, 2);
                Rapp(k, :)=R(k, :)+sigma*randn(1, 2);
                if k12*dt>rand
                    S(k, 1)=2;
                else
                    S(k, 1)=1;
                end              
            else % state=2
                R(k, :)=R(k-1, :)+sqrt(2.0*D2*dt)*randn(1, 2);
                Rapp(k, :)=R(k, :)+sigma*randn(1, 2); 
                if k21*dt>rand
                    S(k, 1)=1;
                else
                    S(k, 1)=2;
                end              
            end          
        end
        A{1, i}=Rapp;
    end
    
%For real trajectories, when traj is inputted
elseif size(traj,1)>2
    disp('traj was inputted');
    %Take out all trajectories that are shorter than 10 steps (11 positions)
    num_traj = max(traj(:,7));
    traj_long = [];
    PID=unique(traj(:,7));
    PID=setdiff(PID,0);
    for a = 1:length(PID)    
        Ppos=traj(:,7)==PID(a);
        traj_part = traj(Ppos,:);
        if size(traj_part,1) >= 11 
            traj_long = cat(1,traj_long,traj_part);
        end
        clear traj_part
    end
    %Find the new PIDs and number of trajectories
    new_PIDs = unique((traj_long(:,7)));
    num_traj_long = length(new_PIDs)
        
    %Save each trajectory into its own cell
    A = {num_traj_long}; % trajectories of X molecules, each cell will contain a 2 column vector
                           % with x and y position data in microns
    for b = 1:num_traj_long %save x and y data to the cell array A, convert to microns
        Ppos=traj_long(:,7)==new_PIDs(b);
        traj_part = traj_long(Ppos,:);
        % pixels to microns from pixel size and magnification of the
        % microscope
        A{1,b} = cat(2, traj_part(:,1)*16/magnif, traj_part(:,2)*16/magnif); 
        clear traj_part;
    end
    dt=xposure/1000; %[sec]% time interval between two neighboring positions
    X = size(A,2); % number of trajectories 
    F = max(traj(:,8)); %find the longest trajectory, save under F for the max number of time steps
    T(1, :)=0:dt:dt*F; %[sec]% time along the trajectory

else
    disp('A was inputted');
    A = traj;
    dt=1/30; %[sec]% time interval between two neighboring positions
    F=200; % number of time steps
    T(1, :)=0:dt:dt*F; %[sec]% time along the trajectory
    X = size(A,2);
end


%% estimation of measurement error
% Standard deviation (SD) of the error arise in position measurement is estimated 
% from MSD-Dt plot, which follows MSD(Dt)=4*D*Dt+4*SD^2.

% calculate squared displacement during Dt=1*dt~10*dt for each molecule
B={1, X};
t=10;
Dt(1, :)=dt:dt:dt*t;
for i=1:X %loop through one trajectory at a time
    if nargin >= 1, F = size(A{1,i},1); F_list(i) = F; end %find the length of the current traj if real data was inputted
    SD_mol=zeros(F-1,t); %make SD_mol large enough for the length of sim trajs    
    for n=1:t 
        for k=1:F-n 
            SD_mol(k, n)=(A{1, i}(k+n, 1)-A{1, i}(k, 1))^2+(A{1, i}(k+n, 2)-A{1, i}(k, 2))^2;
        end  
    end
    B{1, i}=SD_mol;    
    clear SD_mol  
end

% calculate a mean of squared displacement for each molecule
MSD_mol=zeros(t, X);
if nargin == 0, N=F-1:-1:1; end %for simulated data with a constant length
for i=1:X
    if nargin >= 1, F = F_list(i); N=F-1:-1:1; end %for real data with varying length
    for n=1:t 
        temp = [];
        temp(:, 1)=B{1, i}(:, n);
        MSD_mol(n, i)=sum(temp)/N(1, n);
    end
end

% obtain an ensemble average of MSD(Dt)
MSD=zeros(t, 1);
for n=1:t
    MSD(n, 1)=sum(MSD_mol(n,:))/X;
end

% fitting of MSD(Dt) to estimate the standard deviation of the error  
%parameters0 = [1;1]; % Starting guess for slope and y-intercept
parameters0 = [.1;.001];
ub=[inf;inf];
lb=[0;0];
options=optimset('MaxFunEvals',1000,...
    'MaxIter',10000,...
    'TolFun',1e-8,...
    'TolX',1e-7,...
    'Display','iter');
disp('-->MSDFunction')
[parametersMSD,resnormMSD] = lsqcurvefit(@modelMSDFunction,parameters0,...
    Dt',MSD,...
    lb,ub,options);

% display fitting result
figure,plot(Dt, modelMSDFunction(parametersMSD,Dt), 'r');
hold on;
plot(Dt,MSD,'+');
if nargin == 0 
    legend('fitting','simulation');
elseif nargin >=1
    legend('fitting','real data');
end
xlabel('dt (seconds)');
ylabel('Mean square displacement (all molecules, microns^2)');
hold off;
MSD_results = [];
MSD_results = [Dt', MSD, (modelMSDFunction(parametersMSD,Dt))'];

%% estimation of the number of states with different diffusion coefficient
% Based on the probability density function of displacement obtained 
% by assuming different number of diffusion coefficients ranging from 1 to 4,
% the most likely assumption is seeked by referring AIC.
% Displacement should be measured between two positions with the shortest 
% time interval of the trajectory.

% measure displacement
t=1;
Q = [];
for i=1:X
    if nargin >= 1, F = F_list(i); end
%    f=(F-t)*(i-1)+1;
%    g=(F-t)*i;
%    Q(f:g, 1)=sqrt(B{1, i}(:, 1));
    Q = cat(1,Q,sqrt(B{1,i}(:,1)));
end
Q(1,2)=parametersMSD(2);
Q(2,2)=Dt(t);
Q(3,2)=(F-t)*X; %(length of traj-t)*number of trajs = total number of steps
if nargin >= 1, Q(3,2) = sum(F_list-t); end %for real trajectories

% MLE assuming single state
parameters0 = 1;
%parameters0 = 0.1; % Starting guess for diffusion coefficient, adjust as
%needed
%parameters0 = 0.01;
lb = 0;
ub = inf;
options=optimset('MaxFunEvals',1000,...
        'MaxIter',10000,...
        'TolFun',1e-15,...
        'TolX',1e-10,...
        'Display','iter');%,'Algorithm','interior-point');
disp('-->MLE single state')
[parameters1s,fval1s]=fmincon(@(parameters1s)model1stateFunction(parameters1s,Q),...
        parameters0,[],[],[],[],lb,ub,[],options);

% MLE assuming two states
parameters0 = [2;3;1];
%parameters0 = [0.00001;1;1];
%parameters0 = [0.02;0.03;0.1]; % Starting guess for diffusion coefficients
%and the ratio, adjust as needed
lb = [0;0;0];
ub = [inf;inf;1];
%ub = [0.2;0.2;1];
options=optimset('MaxFunEvals',1000,...
        'MaxIter',10000,...
        'TolFun',1e-15,...
        'TolX',1e-10,...
        'Display','iter','Algorithm','interior-point');
    disp('-->MLE two states')
[parameters2s,fval2s]=fmincon(@(parameters2s)model2stateFunction(parameters2s,Q),...
        parameters0,[],[],[],[],lb,ub,[],options);

% MLE assuming three states
%parameters0 = [2;3;4;1;1;1];
parameters0 = [0.02;0.3;1;1;0.1;1];
%parameters0 = [0.00002;1;0.04;3;0.1;0.1]; % Starting guess for diffusion
%coefficients and the ratios, adjust as needed 
lb = [0;0;0;0;0;0];
ub = [inf;inf;inf;inf;inf;inf];
options=optimset('MaxFunEvals',1000,...
        'MaxIter',10000,...
        'TolFun',1e-15,...
        'TolX',1e-10,...
        'Display','iter');%,'Algorithm','interior-point');
    disp('-->MLE three states')
[parameters3s,fval3s]=fmincon(@(parameters3s)model3stateFunction(parameters3s,Q),...
        parameters0,[],[],[],[],lb,ub,[],options);
    
% MLE assuming four states
%parameters0 = [1;2;3;5;4;1;1;4];
%parameters0 = [0.00001;2;0.3;0.5;4;0.1;1;4];
parameters0 = [0.01;0.2;0.03;0.05;4;1;1;4]; % Starting guess for diffusion coefficients and the ratios, adjust as needed
lb = [0;0;0;0;0;0;0;0];
ub = [inf;inf;inf;inf;inf;inf;inf;inf];
options=optimset('MaxFunEvals',1000,...
        'MaxIter',10000,...
        'TolFun',1e-15,...
        'TolX',1e-10,...
        'Display','iter');%,'Algorithm','interior-point');
    disp('-->MLE four states')
[parameters4s,fval4s]=fmincon(@(parameters4s)model4stateFunction(parameters4s,Q),...
        parameters0,[],[],[],[],lb,ub,[],options);

% histogram of displacement
dr = 0.005; % size of bin in microns
r_max= 1.5; % max value of displacement in the histogram
r=dr/2:dr:r_max; %[micrometer]% displacement
temp2(1, :)=hist(Q(:,1), r); %hist of step sizes
PDF=temp2/Q(3,2)/dr; % histogram of displacement from trajectories, normalized by total # of steps and bin size
PDF1s=1/(2*parameters1s(1)*Q(2,2)+2*parametersMSD(2))*r.*exp(-r.*r/...
    (4*parameters1s(1)*Q(2,2)+4*parametersMSD(2)));
PDF2s=parameters2s(3)/(2*parameters2s(1)*Q(2,2)+2*parametersMSD(2))*...
    r.*exp(-r.*r/(4*parameters2s(1)*Q(2,2)+4*parametersMSD(2)))...
    +(1-parameters2s(3))/(2*parameters2s(2)*Q(2,2)+2*parametersMSD(2))*...
    r.*exp(-r.*r/(4*parameters2s(2)*Q(2,2)+4*parametersMSD(2)));
PDF3s=(parameters3s(4)/(2*parameters3s(1)*Q(2,2)+2*parametersMSD(2))*...
    r.*exp(-r.*r/(4*parameters3s(1)*Q(2,2)+4*parametersMSD(2)))...
    +parameters3s(5)/(2*parameters3s(2)*Q(2,2)+2*parametersMSD(2))*...
    r.*exp(-r.*r/(4*parameters3s(2)*Q(2,2)+4*parametersMSD(2)))...
    +parameters3s(6)/(2*parameters3s(3)*Q(2,2)+2*parametersMSD(2))*...
    r.*exp(-r.*r/(4*parameters3s(3)*Q(2,2)+4*parametersMSD(2))))...
    /(parameters3s(4)+parameters3s(5)+parameters3s(6));
    
% display MLE results
figure,semilogx(r,PDF1s,'c');
hold on;
plot(r,PDF2s,'r');
plot(r,PDF3s,'b');
plot(r,PDF,'o');
legend('MLE (1 state)','MLE (2 states)','MLE (3 states)','data');
xlabel('Displacement (microns)');
ylabel('Probability');
hold off;

PDF_results = [];
PDF_results = [r', PDF', PDF1s', PDF2s', PDF3s'];

% calculate AIC
P=[1;3;5;7];
AIC(1,1)=2*fval1s*Q(3,2)+2*P(1,1);
AIC(2,1)=2*fval2s*Q(3,2)+2*P(2,1);
AIC(3,1)=2*fval3s*Q(3,2)+2*P(3,1);
AIC(4,1)=2*fval4s*Q(3,2)+2*P(4,1);

% display AIC
figure,plot(AIC, 'b');
hold on;
xlabel('Number of states');
ylabel('AIC');
hold off;

%% examination of state transition
% The autocorrelation of the squared displacement displays a delta-correlated 
% behavior or an exponential decay in the absence or presence of state transition, respectively. 

% a time series of squared displacement in x-direction for each molecule
C={1, X};
t=1; %[steps]% time interval of displacement measurement
for i=1:X
    if nargin >= 1, F = F_list(i); end
    dx2_mol=zeros(F-t,1);
    for k=1:F-t  
        dx2_mol(k, 1)=(A{1, i}(k+t, 1)-A{1, i}(k, 1))^2;
    end
    C{1, i}=dx2_mol; 
end
clear dx2_mol;

% calculate the autocorrelation for each molecule
if nargin >= 1, F = max(F_list); end
lagT_max=F-t-1; % maximum of lag time [steps] in calculating autocorrelation
AC_dx2_mol=zeros(X,lagT_max+1);
for i=1:X 
    if nargin >= 1, F = F_list(i); end
    AC_dx2_mol_temp=zeros(F-t,F-t);
    for lagT=0:lagT_max % lag time [steps] ranging from 0 to lagT_max
        for k=1:F-t-lagT
            AC_dx2_mol_temp(k, lagT+1)=C{1, i}(k, 1)*C{1, i}(k+lagT, 1);
        end
    end
    for l=1:F-t
        AC_dx2_mol(i, l)=sum(AC_dx2_mol_temp(:, l))/(F-l); 
    end
end
clear AC_dx2_mol_temp;
 
% calculate an ensemble average of the autocorrelation
AC_dx2=zeros(1,lagT_max+1);
if nargin >= 1, F = max(F_list); end
for l=1:F-t
    AC_dx2(1, l)=sum(AC_dx2_mol(:, l))/X;
end

% fit to an exponential function to estimate the decay rate
AC_dx2_fit=AC_dx2/dt/dt;
parameters0 = [1;1;0.01]; % Starting guess, adjust as needed
%parameters0 = [0.0000001;0.000001;0.01];
ub=[inf;inf;inf];
lb=[0;0;0];
options=optimset('MaxFunEvals',100000,...
    'MaxIter',10000,...
    'TolFun',1e-8,...
    'TolX',1e-7,...
    'Display','iter');

max_index = round(length(T)/3*2);
disp('-->ACFunction')
[parametersAC,resnormAC] = lsqcurvefit(@modelACFunction,parameters0,...
    T(1,3:max_index),AC_dx2_fit(1,3:max_index),...
    lb,ub,options);

% display fitting result
AC_results = [];
AC_results = [(T(1,3:max_index))',AC_dx2_fit(1,3:max_index)', (modelACFunction(parametersAC, T(1,3:max_index)))'];
figure,semilogy(T(1,3:max_index), modelACFunction(parametersAC, T(1,3:max_index)), 'b');
hold on;
semilogy(T(1,3:max_index),AC_dx2_fit(1,3:max_index),'r');
legend('fitting','simulation');
xlabel('Lag time');
ylabel('Autocorrelation of dx2 (x1000)');
hold off;

